// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.codeInspection.duplicateExpressions;

import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.UserDataHolder;
import com.intellij.psi.PsiCodeBlock;
import com.intellij.psi.PsiExpression;
import com.intellij.psi.controlFlow.ControlFlowUtil;
import com.intellij.util.ObjectUtils;
import gnu.trove.THashMap;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;

/**
 * @author Pavel.Dolgov
 */
class DuplicateExpressionsContext {
  private static final Key<Map<PsiCodeBlock, DuplicateExpressionsContext>> CONTEXTS_KEY = Key.create("DuplicateExpressionsContext");

  private final Map<PsiExpression, List<PsiExpression>> myOccurrences = new THashMap<>(new ExpressionHashingStrategy());
  private final ComplexityCalculator myComplexityCalculator = new ComplexityCalculator();
  private final SideEffectCalculator mySideEffectCalculator = new SideEffectCalculator();

  void addOccurrence(PsiExpression expression) {
    List<PsiExpression> list = myOccurrences.computeIfAbsent(expression, unused -> new ArrayList<>());
    list.add(expression);
  }

  void forEach(BiConsumer<PsiExpression, List<PsiExpression>> consumer) {
    myOccurrences.forEach(consumer);
  }

  int getComplexity(PsiExpression expression) {
    return myComplexityCalculator.getComplexity(expression);
  }

  boolean mayHaveSideEffect(PsiExpression expression) {
    return mySideEffectCalculator.mayHaveSideEffect(expression);
  }

  @Nullable
  static DuplicateExpressionsContext getOrCreateContext(@NotNull PsiExpression expression, @NotNull UserDataHolder session) {
    PsiCodeBlock nearestBody = findNearestBody(expression);
    if (nearestBody != null) {
      Map<PsiCodeBlock, DuplicateExpressionsContext> contexts = session.getUserData(CONTEXTS_KEY);
      if (contexts == null) {
        session.putUserData(CONTEXTS_KEY, contexts = new THashMap<>());
      }
      return contexts.computeIfAbsent(nearestBody, unused -> new DuplicateExpressionsContext());
    }
    return null;
  }

  @Nullable
  static DuplicateExpressionsContext getContext(@Nullable PsiCodeBlock body, @NotNull UserDataHolder session) {
    Map<PsiCodeBlock, DuplicateExpressionsContext> contexts = session.getUserData(CONTEXTS_KEY);
    return contexts != null ? contexts.get(body) : null;
  }

  static PsiCodeBlock findNearestBody(@NotNull PsiExpression expression) {
    return ObjectUtils.tryCast(ControlFlowUtil.findCodeFragment(expression), PsiCodeBlock.class);
  }
}
